/*
 * Copyright 2016 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

//
// The unittest header must be included before va_x11.h (which might be included
// indirectly).  The va_x11.h includes Xlib.h and X.h.  And the X headers
// define 'Bool' and 'None' preprocessor types.  Gtest uses the same names
// to define some struct placeholders.  Thus, this creates a compile conflict
// if X defines them before gtest.  Hence, the include order requirement here
// is the only fix for this right now.
//
// See bug filed on gtest at https://github.com/google/googletest/issues/371
// for more details.
//
#include "common/factory_unittest.h"

// primary header
#include "vaapiDecoderJPEG.h"

// library headers
#include "common/Array.h"

// system headers
#include <algorithm>

namespace YamiMediaCodec {

const static std::array<uint8_t, 844> g_Simple444P = {
    0xff, 0xd8, 0xff, 0xe0, 0x00, 0x10, 0x4a, 0x46, 0x49, 0x46, 0x00, 0x01,
    0x01, 0x01, 0x00, 0x48, 0x00, 0x48, 0x00, 0x00, 0xff, 0xdb, 0x00, 0x43,
    0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0xff, 0xdb, 0x00, 0x43, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0xff, 0xc0, 0x00, 0x11, 0x08, 0x00, 0x0a, 0x00, 0x0a, 0x03,
    0x01, 0x11, 0x00, 0x02, 0x11, 0x01, 0x03, 0x11, 0x01, 0xff, 0xc4, 0x00,
    0x1f, 0x00, 0x00, 0x01, 0x05, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
    0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0xff, 0xc4, 0x00, 0xb5, 0x10, 0x00,
    0x02, 0x01, 0x03, 0x03, 0x02, 0x04, 0x03, 0x05, 0x05, 0x04, 0x04, 0x00,
    0x00, 0x01, 0x7d, 0x01, 0x02, 0x03, 0x00, 0x04, 0x11, 0x05, 0x12, 0x21,
    0x31, 0x41, 0x06, 0x13, 0x51, 0x61, 0x07, 0x22, 0x71, 0x14, 0x32, 0x81,
    0x91, 0xa1, 0x08, 0x23, 0x42, 0xb1, 0xc1, 0x15, 0x52, 0xd1, 0xf0, 0x24,
    0x33, 0x62, 0x72, 0x82, 0x09, 0x0a, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x25,
    0x26, 0x27, 0x28, 0x29, 0x2a, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a,
    0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x53, 0x54, 0x55, 0x56,
    0x57, 0x58, 0x59, 0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a,
    0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x83, 0x84, 0x85, 0x86,
    0x87, 0x88, 0x89, 0x8a, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9a, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3,
    0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6,
    0xc7, 0xc8, 0xc9, 0xca, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9,
    0xda, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xf1,
    0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xff, 0xc4, 0x00,
    0x1f, 0x01, 0x00, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
    0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0xff, 0xc4, 0x00, 0xb5, 0x11, 0x00,
    0x02, 0x01, 0x02, 0x04, 0x04, 0x03, 0x04, 0x07, 0x05, 0x04, 0x04, 0x00,
    0x01, 0x02, 0x77, 0x00, 0x01, 0x02, 0x03, 0x11, 0x04, 0x05, 0x21, 0x31,
    0x06, 0x12, 0x41, 0x51, 0x07, 0x61, 0x71, 0x13, 0x22, 0x32, 0x81, 0x08,
    0x14, 0x42, 0x91, 0xa1, 0xb1, 0xc1, 0x09, 0x23, 0x33, 0x52, 0xf0, 0x15,
    0x62, 0x72, 0xd1, 0x0a, 0x16, 0x24, 0x34, 0xe1, 0x25, 0xf1, 0x17, 0x18,
    0x19, 0x1a, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x35, 0x36, 0x37, 0x38, 0x39,
    0x3a, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x53, 0x54, 0x55,
    0x56, 0x57, 0x58, 0x59, 0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69,
    0x6a, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x82, 0x83, 0x84,
    0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
    0x98, 0x99, 0x9a, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa,
    0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3, 0xc4,
    0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
    0xd8, 0xd9, 0xda, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea,
    0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xff, 0xda, 0x00,
    0x0c, 0x03, 0x01, 0x00, 0x02, 0x11, 0x03, 0x11, 0x00, 0x3f, 0x00, 0xfe,
    0x7b, 0xb4, 0xcd, 0x33, 0xee, 0xfc, 0xa7, 0x9c, 0x76, 0xfc, 0x7b, 0x8f,
    0xc3, 0x1f, 0xd7, 0x9a, 0xff, 0x00, 0x72, 0x7e, 0x90, 0xbf, 0x48, 0x3b,
    0xbc, 0x77, 0xfb, 0x6a, 0xff, 0x00, 0x97, 0x9b, 0x55, 0xd9, 0x59, 0xeb,
    0xf1, 0x34, 0x7f, 0x56, 0x7d, 0x1c, 0xbe, 0x8e, 0x97, 0xfa, 0x82, 0xfa,
    0x87, 0x4a, 0x7f, 0xf2, 0xeb, 0xc9, 0x2e, 0xdf, 0x77, 0xcb, 0xa6, 0xaf,
    0xad, 0x5d, 0x30, 0xe0, 0x7c, 0xa3, 0xa0, 0xee, 0x3d, 0x3e, 0xb5, 0xfe,
    0x62, 0x62, 0xbe, 0x90, 0xd1, 0xfa, 0xd6, 0x27, 0xfd, 0xb7, 0xfe, 0x62,
    0x2b, 0x7f, 0xcb, 0xc7, 0xff, 0x00, 0x3f, 0x24, 0x7f, 0xa9, 0x98, 0x5f,
    0xa3, 0xa4, 0x7e, 0xab, 0x86, 0xff, 0x00, 0x62, 0x5f, 0xee, 0xf4, 0x7f,
    0xe5, 0xd2, 0xff, 0x00, 0x9f, 0x71, 0xfe, 0xe8, 0x69, 0x8a, 0xbc, 0x70,
    0x3a, 0x0e, 0xc3, 0xd4, 0x55, 0xfd, 0x21, 0x31, 0x58, 0xaf, 0xf6, 0xef,
    0xf6, 0x9a, 0xff, 0x00, 0x15, 0x5f, 0xf9, 0x7d, 0x53, 0xfb, 0xdf, 0xde,
    0x23, 0xe8, 0xe3, 0x84, 0xc2, 0xdb, 0x2f, 0xff, 0x00, 0x66, 0xc3, 0xef,
    0x4b, 0xfe, 0x5c, 0xd3, 0xf2, 0xfe, 0xe9, 0xd5, 0xed, 0x5f, 0x41, 0xf9,
    0x0a, 0xff, 0x00, 0x31, 0x31, 0x58, 0xac, 0x4f, 0xd6, 0x71, 0x1f, 0xed,
    0x15, 0xff, 0x00, 0x8f, 0x5b, 0xfe, 0x5f, 0x54, 0xff, 0x00, 0x9f, 0x92,
    0xfe, 0xf1, 0xfe, 0xa5, 0xe1, 0x30, 0x98, 0x5f, 0xaa, 0xe1, 0xbf, 0xd9,
    0xb0, 0xff, 0x00, 0xee, 0xf4, 0x7f, 0xe5, 0xcd, 0x3f, 0xf9, 0xf7, 0x1f,
    0xee, 0x9f, 0xff, 0xd9
};

const static std::array<uint8_t, 424> g_Simple422V = {
    0xff, 0xd8, 0xff, 0xe0, 0x00, 0x10, 0x4a, 0x46, 0x49, 0x46, 0x00, 0x01,
    0x01, 0x01, 0x00, 0x48, 0x00, 0x48, 0x00, 0x00, 0xff, 0xdb, 0x00, 0x43,
    0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0xff, 0xdb, 0x00, 0x43, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0xff, 0xc0, 0x00, 0x11, 0x08, 0x00, 0x0a, 0x00, 0x0a, 0x03,
    0x01, 0x22, 0x00, 0x02, 0x21, 0x01, 0x03, 0x21, 0x01, 0xff, 0xc4, 0x00,
    0x16, 0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x0a, 0xff, 0xc4, 0x00,
    0x23, 0x10, 0x00, 0x01, 0x03, 0x03, 0x02, 0x07, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x05, 0x11, 0x21, 0x41,
    0x51, 0x02, 0x31, 0x22, 0x32, 0x61, 0x71, 0x81, 0x91, 0xa1, 0xff, 0xc4,
    0x00, 0x14, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0xff, 0xc4, 0x00, 0x2a,
    0x11, 0x00, 0x00, 0x03, 0x04, 0x08, 0x07, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x03, 0x04, 0x00, 0x07, 0x08, 0x21,
    0x02, 0x11, 0x13, 0x14, 0x44, 0x51, 0x53, 0x63, 0x23, 0x24, 0x31, 0x34,
    0x62, 0x71, 0x73, 0xd1, 0xff, 0xda, 0x00, 0x0c, 0x03, 0x01, 0x00, 0x02,
    0x11, 0x03, 0x11, 0x00, 0x3f, 0x00, 0xcf, 0x73, 0x63, 0x67, 0x2c, 0x69,
    0x35, 0x8b, 0x79, 0xb8, 0x9e, 0x91, 0xf2, 0x6a, 0x96, 0x86, 0xc3, 0x03,
    0x84, 0x6c, 0x2e, 0x31, 0xdd, 0x4d, 0x80, 0x40, 0xa0, 0xd8, 0x58, 0x64,
    0x25, 0x70, 0x30, 0x3d, 0x04, 0x9d, 0x44, 0x24, 0x4f, 0x3c, 0x1e, 0x7a,
    0xb3, 0x6b, 0x99, 0x81, 0xdd, 0x1b, 0xe5, 0xb7, 0xf9, 0xed, 0x8c, 0x55,
    0x51, 0x3e, 0xf0, 0x85, 0x52, 0x91, 0xb5, 0x09, 0xa8, 0x3b, 0x12, 0x66,
    0xa5, 0x2d, 0xb6, 0x51, 0x61, 0xc6, 0x16, 0x5d, 0xd8, 0xdc, 0x06, 0xca,
    0x5c, 0x20, 0xaa, 0xe8, 0x56, 0x5f, 0x4c, 0xa5, 0xd1, 0x94, 0xb4, 0x90,
    0xb2, 0xee, 0x85, 0x2a, 0x61, 0xb1, 0xc3, 0x93, 0x85, 0x2f, 0x4e, 0x8e,
    0xeb, 0x7f, 0xff, 0xd9
};

const static std::array<uint8_t, 423> g_Simple422H = {
    0xff, 0xd8, 0xff, 0xe0, 0x00, 0x10, 0x4a, 0x46, 0x49, 0x46, 0x00, 0x01,
    0x01, 0x01, 0x00, 0x48, 0x00, 0x48, 0x00, 0x00, 0xff, 0xdb, 0x00, 0x43,
    0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0xff, 0xdb, 0x00, 0x43, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0xff, 0xc0, 0x00, 0x11, 0x08, 0x00, 0x0a, 0x00, 0x0a, 0x03,
    0x01, 0x22, 0x00, 0x02, 0x12, 0x01, 0x03, 0x12, 0x01, 0xff, 0xc4, 0x00,
    0x16, 0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x0a, 0xff, 0xc4, 0x00,
    0x23, 0x10, 0x00, 0x01, 0x03, 0x03, 0x02, 0x07, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x05, 0x11, 0x21, 0x41,
    0x51, 0x02, 0x31, 0x22, 0x32, 0x61, 0x71, 0x81, 0x91, 0xa1, 0xff, 0xc4,
    0x00, 0x14, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0xff, 0xc4, 0x00, 0x2d,
    0x11, 0x00, 0x00, 0x03, 0x04, 0x04, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x04, 0x08, 0x00, 0x02, 0x03, 0x43,
    0x11, 0x17, 0x24, 0x62, 0x06, 0x16, 0x23, 0x31, 0x32, 0x33, 0x51, 0x52,
    0x53, 0x61, 0x63, 0x71, 0x91, 0xb1, 0xd1, 0xff, 0xda, 0x00, 0x0c, 0x03,
    0x01, 0x00, 0x02, 0x11, 0x03, 0x11, 0x00, 0x3f, 0x00, 0xcf, 0x73, 0x63,
    0x67, 0x2c, 0x69, 0x35, 0x8b, 0x79, 0xb8, 0x9e, 0x91, 0xf2, 0x6a, 0x96,
    0x86, 0xc3, 0x03, 0x84, 0x6c, 0x2e, 0x31, 0xdd, 0x4d, 0x80, 0x40, 0xa0,
    0xd8, 0x58, 0x64, 0x25, 0x70, 0x30, 0x3d, 0x04, 0x95, 0xaa, 0xc5, 0x0a,
    0x01, 0x87, 0x6f, 0xd2, 0x77, 0x3c, 0x43, 0x33, 0x47, 0x6b, 0x9c, 0xc7,
    0xdd, 0x1e, 0x18, 0xc8, 0x55, 0xa6, 0x8c, 0xe3, 0xe0, 0xda, 0x23, 0xe9,
    0x9a, 0x9b, 0x13, 0x7a, 0x1d, 0xe6, 0x54, 0x92, 0x4a, 0x74, 0x0a, 0xbf,
    0x72, 0xc2, 0x01, 0x93, 0x2d, 0x2b, 0xa7, 0xd8, 0x7e, 0xb2, 0x90, 0x92,
    0x4a, 0x15, 0xab, 0xf7, 0x2c, 0xc5, 0xf5, 0x65, 0x64, 0xc3, 0xe1, 0x8d,
    0xd6, 0xff, 0xd9
};

const static std::array<uint8_t, 383> g_SimpleIMC3 = {
    0xff, 0xd8, 0xff, 0xe0, 0x00, 0x10, 0x4a, 0x46, 0x49, 0x46, 0x00, 0x01,
    0x01, 0x01, 0x00, 0x48, 0x00, 0x48, 0x00, 0x00, 0xff, 0xdb, 0x00, 0x43,
    0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0xff, 0xdb, 0x00, 0x43, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0xff, 0xc0, 0x00, 0x11, 0x08, 0x00, 0x0a, 0x00, 0x0a, 0x03,
    0x01, 0x22, 0x00, 0x02, 0x11, 0x01, 0x03, 0x11, 0x01, 0xff, 0xc4, 0x00,
    0x16, 0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x0a, 0xff, 0xc4, 0x00,
    0x23, 0x10, 0x00, 0x01, 0x03, 0x03, 0x02, 0x07, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x05, 0x11, 0x21, 0x41,
    0x51, 0x02, 0x31, 0x22, 0x32, 0x61, 0x71, 0x81, 0x91, 0xa1, 0xff, 0xc4,
    0x00, 0x15, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x09, 0xff, 0xc4, 0x00,
    0x24, 0x11, 0x00, 0x00, 0x03, 0x06, 0x07, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x03, 0x08, 0x00, 0x02, 0x04,
    0x07, 0x23, 0x43, 0x16, 0x17, 0x31, 0x42, 0x44, 0x52, 0x61, 0x63, 0xff,
    0xda, 0x00, 0x0c, 0x03, 0x01, 0x00, 0x02, 0x11, 0x03, 0x11, 0x00, 0x3f,
    0x00, 0xcf, 0x73, 0x63, 0x67, 0x2c, 0x69, 0x35, 0x8b, 0x79, 0xb8, 0x9e,
    0x91, 0xf2, 0x6a, 0x96, 0x86, 0xc3, 0x03, 0x84, 0x6c, 0x2e, 0x31, 0xdd,
    0x4d, 0x80, 0x40, 0xa0, 0xd8, 0x58, 0x64, 0x25, 0x70, 0x30, 0x3d, 0x04,
    0x9a, 0x55, 0x6a, 0x7e, 0x60, 0xe3, 0xc7, 0xaa, 0xdc, 0x88, 0xd6, 0x28,
    0xc1, 0xde, 0x5f, 0xc8, 0x1a, 0xf5, 0x24, 0x94, 0xb1, 0x2e, 0xf2, 0xfd,
    0xca, 0x56, 0xe1, 0x78, 0x85, 0xf4, 0xf0, 0xd0, 0x6f, 0xff, 0xd9
};

class VaapiDecoderJPEGTest
    : public FactoryTest<IVideoDecoder, VaapiDecoderJPEG>
{
protected:
    /* invoked by gtest before the test */
    virtual void SetUp() {
        return;
    }

    /* invoked by gtest after the test */
    virtual void TearDown() {
        return;
    }
};

#define VAAPIDECODER_JPEG_TEST(name) \
    TEST_F(VaapiDecoderJPEGTest, name)

VAAPIDECODER_JPEG_TEST(Factory)
{
    FactoryKeys mimeTypes;
    mimeTypes.push_back(YAMI_MIME_JPEG);
    doFactoryTest(mimeTypes);
}

class TestDecodeBuffer
    : public VideoDecodeBuffer
{
public:
    typedef SharedPtr<TestDecodeBuffer> Shared;

    template<std::size_t N>
    static Shared create(
        const std::array<uint8_t, N>& data,
        const std::string& fourcc,
        const uint32_t sliceStart)
    {
        Shared buffer(new TestDecodeBuffer(fourcc, sliceStart));
        buffer->data = const_cast<uint8_t*>(data.data());
        buffer->size = data.size();
        buffer->timeStamp = 0;
        return buffer;
    }

    uint32_t getFourcc() const
    {
        return VA_FOURCC(m_fourcc[0], m_fourcc[1], m_fourcc[2], m_fourcc[3]);
    }

    friend std::ostream& operator <<(std::ostream& os, const TestDecodeBuffer& t)
    {
        os << t.m_fourcc;
        return os;
    }

private:
    TestDecodeBuffer(const std::string& fourcc, const uint32_t sliceStart)
        : m_fourcc(fourcc)
        , m_sliceStart(sliceStart)
    { }

    std::string m_fourcc;

public:
    const uint32_t m_sliceStart;
};

class JPEGTest
    : public ::testing::Test
    , public ::testing::WithParamInterface<TestDecodeBuffer::Shared>
{ };

TEST_P(JPEGTest, Decode_Simple)
{
    VaapiDecoderJPEG decoder;
    VideoConfigBuffer config;
    VideoDecodeBuffer buffer = *GetParam();

    config.flag = 0;

    ASSERT_EQ(YAMI_SUCCESS, decoder.start(&config));
    ASSERT_EQ(YAMI_DECODE_FORMAT_CHANGE, decoder.decode(&buffer));
    ASSERT_EQ(YAMI_SUCCESS, decoder.decode(&buffer));
    ASSERT_EQ(YAMI_SUCCESS, decoder.decode(&buffer));

    SharedPtr<VideoFrame> output(decoder.getOutput());
    ASSERT_TRUE(bool(output));
    EXPECT_EQ(GetParam()->getFourcc(), output->fourcc);
}

TEST_P(JPEGTest, Decode_SimpleMulti)
{
    /*
     * Test MJPEG decoding.  VaapiDecoderJPEG::decode will only accept
     * one JPEG buffer at a time (i.e. the buffer must start at one SOI marker
     * and end at one EOI marker.  Otherwise it will indicate YAMI_FAIL.  It
     * is up to the caller to ensure the MJPEG is divided on single JPEG images
     * for each call to decode.
     */
    VaapiDecoderJPEG decoder;
    VideoConfigBuffer config;
    VideoDecodeBuffer buffer;
    SharedPtr<VideoFrame> output;
    const uint8_t *paramData = GetParam()->data;
    const size_t paramSize = GetParam()->size;
    std::vector<uint8_t> mjpeg(paramSize * 2); // 2 images

    // First image
    std::copy(paramData, paramData + paramSize, mjpeg.begin());
    // Second image
    std::copy(paramData, paramData + paramSize, mjpeg.begin() + paramSize);

    config.flag = 0;

    // Set buffer at first jpeg image
    buffer.data = const_cast<uint8_t*>(mjpeg.data());
    buffer.size = paramSize; // Length of first jpeg image data
    buffer.timeStamp = 0;

    ASSERT_EQ(YAMI_SUCCESS, decoder.start(&config));

    // Decode returns format change for the first frame decoding at first time.
    ASSERT_EQ(YAMI_DECODE_FORMAT_CHANGE, decoder.decode(&buffer));

    // Decode has been initiated by the last "decoder.decode(&buffer)" call.
    ASSERT_EQ(YAMI_SUCCESS, decoder.decode(&buffer));

    output = decoder.getOutput();
    ASSERT_TRUE(bool(output));
    EXPECT_EQ(GetParam()->getFourcc(), output->fourcc);

    // Set buffer at second jpeg image
    buffer.data = const_cast<uint8_t*>(mjpeg.data() + paramSize);
    buffer.size = paramSize; // Length of second jpeg image data
    buffer.timeStamp = 1;

    //Decode the seconde jpeg image.
    ASSERT_EQ(YAMI_SUCCESS, decoder.decode(&buffer));

    output = decoder.getOutput();
    ASSERT_TRUE(bool(output));
    EXPECT_EQ(GetParam()->getFourcc(), output->fourcc);

    /* decoder should fail if more than one image in buffer */
    buffer.data = const_cast<uint8_t*>(mjpeg.data());
    buffer.size = mjpeg.size();  // Length of entire mjpeg
    buffer.timeStamp = 3;

    // In this case, the decode should fail since it encounters a second image in the
    // buffer.
    ASSERT_EQ(YAMI_FAIL, decoder.decode(&buffer));
}

TEST_P(JPEGTest, Decode_SimpleTruncated)
{
    const size_t size(GetParam()->size);

    for (size_t i(1); i < size; ++i) {
        VaapiDecoderJPEG decoder;
        VideoConfigBuffer config;
        VideoDecodeBuffer buffer;

        config.flag = 0;

        buffer.data = GetParam()->data;
        buffer.size = i;
        buffer.timeStamp = 0;

        ASSERT_EQ(YAMI_SUCCESS, decoder.start(&config));

        if (i <= GetParam()->m_sliceStart + 1) {
            // decode always return YAMI_FAIL since it doesn't have enough data.
            EXPECT_EQ(YAMI_FAIL, decoder.decode(&buffer)) << i;
            EXPECT_EQ(YAMI_FAIL, decoder.decode(&buffer)) << i;
        } else {
            // The decoder has enough data to produce a slice for encoding
            // but does not encounter an EOI marker, so a fake EOI is inserted.
            // This means the decoder will still attempt to decode the slice
            // but cannot guarantee the final decoded output is correct since
            // the slice may not actually be complete.
            EXPECT_EQ(YAMI_DECODE_FORMAT_CHANGE, decoder.decode(&buffer)) << i;
            EXPECT_EQ(YAMI_SUCCESS, decoder.decode(&buffer)) << i;
        }
    }
}

/** Teach Google Test how to print a TestDecodeBuffer::Shared object */
void PrintTo(const TestDecodeBuffer::Shared& t, std::ostream* os)
{
    *os << *t;
}

INSTANTIATE_TEST_CASE_P(
    VaapiDecoder, JPEGTest,
    ::testing::Values(
        TestDecodeBuffer::create(g_Simple444P, "444P", 623),
        TestDecodeBuffer::create(g_SimpleIMC3, "IMC3", 313),
        TestDecodeBuffer::create(g_Simple422H, "422H", 321),
        TestDecodeBuffer::create(g_Simple422V, "422V", 318)));

}
